#!/bin/bash
#
# This file is submitted to the EGEE broker instead of the real executable.
# It is responsible for copying input files from storage elements to make
# them available localy. After that it prepares and calls the real
# executable.

export JOB_NAME="Job1"
export LCG_GFAL_VO=gilda
export LCG_GFAL_INFOSYS=gilda-bdii.ct.infn.it:2170
export LFC_HOST=lfc-gilda.ct.infn.it
export VO_GILDA_DEFAULT_SE=aliserv6.ct.infn.it
export LCG_CATALOG_TYPE=lfc

LOG=std.log
ERR=std.err

function remove_file()
{
    GUID=`lcg-lg $1 2> /dev/null`
    REPS=`lcg-lr $1 2> /dev/null`
    for i in $REPS; do
        lcg-del $i &> /dev/null
        lcg-uf $GUID $i &> /dev/null
    done
    lcg-del -a $1 &> /dev/null
    lfc-rm ${1/lfn:/} &> /dev/null
    return 0
}

cd `dirname $0`

function dolog()
{
    if [ $# -eq 0 ]; then
	cat >> $LOG
    else
	echo "$@" >> $LOG
    fi
}

function doerr()
{
    if [ $# -eq 0 ]; then
	cat >> $ERR
    else
	echo "$@" >> $ERR
    fi
}

function doexit()
{
    dolog "EXIT CODE:$1"
    exit $1
}

# Check if wrapper script is the first instance or not
[ ! -d info ] && IS_MASTER="y"

rm -f $LOG $ERR
touch $LOG $ERR

dolog "=======" `date` "Execution of fake script starts..."
[ "$IS_MASTER" = "y" ] && tar zxf info.tar.gz

FEXE=`cat info/$JOB_NAME/exe_name`
EXE=`basename $FEXE`
TWFNAME=$(basename $(dirname $(dirname $FEXE)))
WFNAME=${TWFNAME/_files/}
dolog "The real executable is: \"$EXE\""
dolog "The WF name is: \"$WFNAME\""
chmod u+x "$EXE"

dolog "==== Processing files based on info structure ..."
if [ -d info/$JOB_NAME/files -a "$IS_MASTER" = "y" ]; then
    for f in info/$JOB_NAME/files/*; do
        FILE_TYPE=`cat $f/parameters/file_type/value`
        FILE_NAME=`cat $f/name`
        FILE_DIRECTION=`cat $f/direction`
	IS_COPY="no"
	if [ -d $f/parameters/is_copy ]; then
	    [ "1" = "`cat $f/parameters/is_copy/value`" ] && IS_COPY="yes"
	else
	    IS_COPY="yes"
	fi
        if [ "$FILE_DIRECTION" = "input" -a "$FILE_TYPE" = "remote" -a "yes" = "$IS_COPY" ]; then
	    if [ "lfn" = "`echo $FILE_NAME | cut -c-3`" ]; then
		FILE_LOCAL_PATH=`cat $f/parameters/local_path/value`
		FILE_LOCAL_NAME=`basename $FILE_LOCAL_PATH`
                if [ "COLL" = "`cat info/$JOB_NAME/parameters/ps_type/value 2> /dev/null`" -a -d $f/parameters/ps_id ]; then
		    # Copy parametric outputs of eWFs...
		    dolog "We are a collector job!"
		    rm -rf $FILE_LOCAL_NAME.*
		    REMOTE_BN=`basename $FILE_NAME`
		    REMOTE_DN=`dirname $FILE_NAME`
		    lfc-ls ${REMOTE_DN/lfn:/} | dolog
		    FILES=`lfc-ls ${REMOTE_DN/lfn:/} | grep "^$REMOTE_BN\.$WFNAME\."`
		    dolog "Files we use: $FILES"
		    for i in $FILES; do
		        DESTINATION="file:`pwd`/$FILE_LOCAL_NAME.$i"
		        dolog "Copying file $REMOTE_DN/$i to $DESTINATION..."
		        lcg-cp $REMOTE_DN/$i $DESTINATION 2> lcg_cp.err > lcg_cp.out
		        if [ $? != 0 ]; then
		            doerr "XXX_WRAP_ERROR_XXX"
		            doerr "Copying file \"$REMOTE_DN/$i\" from SE failed!"
		            doerr "The following command failed: \"lcg-cp $REMOTE_DN/$i $DESTINATION\""
		            doerr "-- stderr begins --"
		            cat lcg_cp.err | doerr
		            doerr "-- stderr ends --"
		            doerr "-- stdout begins --"
		            cat lcg_cp.out | doerr
		            doerr "-- stdout ends --"
		            rm -f lcg_cp.{out,err}
		            doexit -1
		        fi
		        rm -f lcg_cp.{out,err}
            	    done
                else
	    	    rm -f $FILE_LOCAL_NAME
	    	    DESTINATION="file:`pwd`/$FILE_LOCAL_NAME"
	    	    dolog "lcg-cp $FILE_NAME $DESTINATION"
	    	    lcg-cp $FILE_NAME $DESTINATION 2> lcg_cp.err > lcg_cp.out
	    	    if [ $? != 0 ]; then
	    		doerr "XXX_WRAP_ERROR_XXX"
	    		doerr "Copying file \"$FILE_NAME\" from SE failed!"
	    		doerr "The following command failed: \"lcg-cp $FILE_NAME $DESTINATION\""
	    		doerr "-- stderr begins --"
	    		cat lcg_cp.err | doerr
	    		doerr "-- stderr ends --"
	    		doerr "-- stdout begins --"
	    		cat lcg_cp.out | doerr
	    		doerr "-- stdout ends --"
	    		rm -f lcg_cp.{out,err}
	    		doexit -1
	    	    fi
	    	    rm -f lcg_cp.{out,err}
		fi
	    else
		doerr "XXX_WRAP_ERROR_XXX"
		doerr "Remote input file \"$FILE_NAME\" unsupported!"
		doexit -1
	    fi
	elif [ "$FILE_DIRECTION" = "output" -a "$FILE_TYPE" = "remote" ]; then
	    if [ "lfn" = "`echo $FILE_NAME | cut -c-3`" ]; then
	    	dolog "File \"$FILE_NAME\" is a remote output file, must be removed from the storage before execution of job."
		lfc-ls ${FILE_NAME/lfn:/} &> /dev/null
		if [ $? = 0 ]; then
		    dolog "The file exists, we are about to remove it..."
	    	    dolog "remove_file $FILE_NAME"
	    	    remove_file $FILE_NAME
		fi
	    elif [ "GEN" = "`cat info/$JOB_NAME/parameters/ps_type/value 2> /dev/null`" ]; then
	        rm -rf $(basename `cat $f/parameters/local_path/value`).*
	    else
		doerr "XXX_WRAP_ERROR_XXX"
		doerr "Remote output file \"$FILE_NAME\" unsupported!"
		doexit -1
	    fi
	else
	    dolog "File \"$FILE_NAME\" unhandled."
	fi
    done
fi

dolog "==== File processing done."

if [ -f "$EXE" -a -x "$EXE" ]; then
    dolog "======="
    dolog "Starting the real executable \"$EXE\"..."
    ./$EXE "$@"
    RET=$?
    dolog "Finished with return code: $RET" >>$LOG
else
    dolog "Executable \"$EXE\" does not exist or is not permitted to run."
fi

if [ "GEN" = "`cat info/$JOB_NAME/parameters/ps_type/value 2> /dev/null`" ]; then
    dolog "We're a GEN job, need to upload produced files..."
    if [ -d info/$JOB_NAME/files -a "$IS_MASTER" = "y" ]; then
        for f in info/$JOB_NAME/files/*; do
            FILE_TYPE=`cat $f/parameters/file_type/value`
	    LPATH=$(basename $(cat $f/parameters/local_path/value))
            FILE_NAME=`cat $f/name`
            FILE_DIRECTION=`cat $f/direction`
            if [ "$FILE_DIRECTION" = "output" -a "$FILE_TYPE" = "remote" ]; then
                lfc-mkdir -p $FILE_NAME &> $$.lfc_mkdir.log
                if [ $? != 0 ]; then
		    HAS=`grep "File exists" $$.lfc_mkdir.log`
		    if [ -z "$HAS" ]; then
                	doerr "Failed to create remote output directory!"
                	doerr "Failed command was: \"lfc-mkdir -p $FILE_NAME\""
                	cat $$.lfc_mkdir.log | doerr
                	doexit -1
		    fi
                fi
		lfc-chmod 700 $FILE_NAME &> /dev/null
                FILES=`lfc-ls -l $FILE_NAME 2> /dev/null | grep -E "^\-.* $LPATH\.*" | awk '{print $9}'`
                [ $? = 0 ] && for i in $FILES; do
		    dolog "Removing present output file lfn:$FILE_NAME/$i"
                    remove_file lfn:$FILE_NAME/$i
                done
	        for i in $(basename `cat $f/parameters/local_path/value`).*; do
                    dolog "About to upload file $i..."
                    lcg-cr -l lfn:$FILE_NAME/$i file:`pwd`/$i &> $$.lcg_cr.log
                    if [ $? != 0 ]; then
                        doerr "Failed command: \"lcg-cr -l lfn:$FILE_NAME/$i file:`pwd`/$i\""
                        cat $$.lcg_cr.log | doerr
                        doexit -1
                    fi
                done
            fi
        done
    fi
    doexit $RET
fi

(
[ -d info/$JOB_NAME/files ] && for f in info/$JOB_NAME/files/*; do
    dolog ".... Processing $f"
    unset F_LOCAL_PATH F_LOCAL_PATH_NAME
    F_ID=`basename $f`
    F_FILE_TYPE=`cat $f/parameters/file_type/value`
    if [ -f $f/parameters/local_path/value ]; then
        F_LOCAL_PATH=`cat $f/parameters/local_path/value`
        F_LOCAL_PATH_NAME=`basename $F_LOCAL_PATH`
    fi
    F_ATTTRIBUTE=`cat $f/attribute`
    F_NAME=`cat $f/name`
    F_NAME_DIR=`dirname $F_NAME`
    F_NAME_NAME=`basename $F_NAME`
    F_DIRECTION=`cat $f/direction`
    if [ "$F_DIRECTION" = "output" -a "$F_FILE_TYPE" = "remote" ]; then
        dolog "handle $f: remote output file"
	dolog "Createing directory of remote output file $f..."
	DNAME=`dirname ${F_NAME/lfn:/}`
	lfc-mkdir -p $DNAME &> mkdir.log.$$
	if [ $? != 0 ]; then
	    HAS=`grep "File exists" mkdir.log.$$`
	    if [ -z "$HAS" ]; then
            	doerr "Failed to create remote output directory for file $F_NAME!"
            	doerr "Failed command was: \"lfc-mkdir -p $DNAME\""
            	cat mkdir.log.$$ | doerr
            	doexit -1
	    fi
	fi
	lfc-chmod 700 $DNAME &> /dev/null
        dolog "Executing: \"lcg-cr -l $F_NAME file:`pwd`/$F_LOCAL_PATH_NAME\""
        lcg-cr -l $F_NAME file:`pwd`/$F_LOCAL_PATH_NAME &> $$.lcg_cr.log
        if [ $? != 0 ]; then
            doerr "Creating remote output file \"$F_NAME\" failed!"
            cat $$.lcg_cr.log | doerr
            doexit -1
        fi
    else
        dolog "handle $f: unhandled"
    fi
done
) >> $LOG 2>&1
																			    
dolog "======= Execution of fake script has finished."

doexit $RET
