#!/usr/bin/python
# -*- coding: iso-8859-2 -*-

###########################################################################
# A Simple Programming Language
#
# Copyright (C) 2006-2008 Gerg RDI
# http://cactus.rulez.org/
#
# Published under the terms of the GNU General Public License 2.0
# $Id: sp.py 188 2006-03-12 11:59:46Z cactus $
###########################################################################
#
# SP is a toy language allowing the following simple statements:
#
# READ var              Read an int from the user, and store it in 'var'
# WRITE expr            Print the value of 'expr'
# GOTO lbl [IF expr]    Jump to the specified line. If 'expr' is present,
#                       jump only if the result of 'expr' >= 0
# LET var = expr        Set value of 'var' to 'expr'
#
# There are 33 variables, named X, Y, Z, X0, ..., X9, Y0, ...
# The type of variables and expressions are all 32-bit signed integers
# 
###########################################################################

import sp

class Stmt:
    pass

class Read (Stmt):
    def __init__ (self, var):
        self.var = var
        
    def evaluate (self, interpreter):
        interpreter.set_variable(self.var, interpreter.read ())
        
    def compile_to_asm (self, compiler):
        return ['call _sp_input',
                'mov [%s], edx' % compiler.var_address (self.var)]
        
class Write (Stmt):
    def __init__ (self, expr):
        self.expr = expr
        
    def evaluate (self, interpreter):
        interpreter.write(self.expr.evaluate(interpreter))

    def compile_to_asm (self, compiler):
        return self.expr.push_to_stack (compiler) + \
               ['push dword _sp_fmt',
                'call printf',
                'sub esp, 8']

class Assign (Stmt):
    def __init__(self, var, expr):
        self.var = var
        self.expr = expr

    def evaluate(self, interpreter):
        interpreter.set_variable(self.var, self.expr.evaluate(interpreter))

    def compile_to_asm (self, compiler):
        expr = self.expr.const_fold (compiler.get_evaluator ())
        
        return expr.push_to_stack (compiler) + \
               ['pop edx',
                'mov [%s], edx' % compiler.var_address (self.var)]

class Goto (Stmt):
    def __init__ (self, lbl, cond = None):
        self.lbl = lbl
        self.cond = cond

    def evaluate (self, interpreter):
        if not self.cond or self.cond.evaluate(interpreter) >= 0:
            interpreter.jump (self.lbl)

    def compile_to_asm (self, compiler):
        if self.cond:
            return self.cond.push_to_stack (compiler) + \
                   ['pop ecx',
                    'cmp ecx, 0',
                    'jge %s' % compiler.label (self.lbl)]
        else:
            return ['jmp %s' % compiler.label (self.lbl)]
           
