#!/usr/bin/python
# -*- coding: iso-8859-2 -*-

###########################################################################
# Parser for a Simple Programming Language
#
# Copyright (C) 2006-2008 Gerg RDI
# http://cactus.rulez.org/
#
# Published under the terms of the GNU General Public License 2.0
# $Id: sp.py 188 2006-03-12 11:59:46Z cactus $
###########################################################################
#
# SP is a toy language allowing the following simple statements:
#
# READ var              Read an int from the user, and store it in 'var'
# WRITE expr            Print the value of 'expr'
# GOTO lbl [IF expr]    Jump to the specified line. If 'expr' is present,
#                       jump only if the result of 'expr' >= 0
# LET var = expr        Set value of 'var' to 'expr'
#
# There are 33 variables, named X, Y, Z, X0, ..., X9, Y0, ...
# The type of variables and expressions are all 32-bit signed integers
# 
###########################################################################

import stmt
import expr

########################        
# Create lexer
########################        

def token(name, fun = lambda x: x):
    return lambda x: (name, fun(x))


lexer = [('\s+',          ''),
         ('[+\-*/:;()=]', lambda x: (x, x)),
         
         ('(?i)GOTO',     token('GOTO')),
         ('(?i)IF',       token('IF')),
         ('(?i)READ',     token('READ')),
         ('(?i)WRITE',    token('WRITE')),
         ('(?i)LET',      token('LET')),
         
         ('(?i)[a-z]',    token('LETTER', lambda x: x.lower())),
         ('(?i)[0-9]',    token('DIGIT',  lambda x: int(x)))]


########################        
# Create parser
########################        

grammar = [('program', ['progline'],                      lambda (l,), c:      [l]),
           ('program', ['program', 'progline'],           lambda (prog, l), c: prog + [l]),
           
           ('progline', ['number', ':', 'stmt', ';'],     lambda (lbl, col, stmt, scol), c: (lbl, stmt)),
           ('progline', ['stmt', ';'],                    lambda (stmt, scol), c:           (None, stmt)),
           
           ('stmt',   ['READ', 'varname'],                lambda (read, id), c:             stmt.Read (id)),
           ('stmt',   ['WRITE', 'expr'],                  lambda (write, expr), c:          stmt.Write (expr)),
           ('stmt',   ['LET', 'varname', '=', 'expr'],    lambda (let, id, eq, expr), c:    stmt.Assign (id, expr)),
           ('stmt',   ['GOTO', 'number'],                 lambda (goto, lbl), c:            stmt.Goto (lbl)),
           ('stmt',   ['GOTO', 'number', 'IF', 'expr'],   lambda (goto, lbl, if_, cond), c: stmt.Goto (lbl, cond)),
           
           ('expr',   ['term'],                           lambda (t,),c:        t),
           ('expr',   ['term', '+', 'term'],              lambda (a, op, b), c: expr.Plus (a, b)),
           ('expr',   ['term', '-', 'term'],              lambda (a, op, b), c: expr.Minus (a, b)),
           
           ('term',   ['factor'],                         lambda (f,),c:        f),
           ('term',   ['factor', '*', 'factor'],          lambda (a, op, b), c: expr.Mul (a, b)),
           ('term',   ['factor', '/', 'factor'],          lambda (a, op, b), c: expr.Div (a, b)),
           
           ('factor', ['(', 'expr', ')'],                 lambda (lp, e, rp),c: e),
           ('factor', ['number'],                         lambda (num,),c:      expr.NumLit (num)),
           ('factor', ['varname'],                        lambda (id,),c:       expr.VariableVal (id)),
           
           ('varname', ['LETTER'],                        lambda (l,), c:        l),
           ('varname', ['LETTER', 'DIGIT'],               lambda (l, d), c:     '%s%d' % (l, d)),
           
           ('number', ['DIGIT'],                          lambda (d,),c:     d),
           ('number', ['number', 'DIGIT'],                lambda (num, d),c: num * 10 + d)]

           
import yappy.parser
parser = yappy.parser.Yappy (lexer, grammar)

def load_sp (filename):
    f = open (filename, 'r')
    return parser.input (f.read ())
