#!/usr/bin/python
# -*- coding: iso-8859-2 -*-

###########################################################################
# A Simple Programming Language
#
# Copyright (C) 2006-2008 Gerg RDI
# http://cactus.rulez.org/
#
# Published under the terms of the GNU General Public License 2.0
# $Id: sp.py 188 2006-03-12 11:59:46Z cactus $
###########################################################################
#
# SP is a toy language allowing the following simple statements:
#
# READ var              Read an int from the user, and store it in 'var'
# WRITE expr            Print the value of 'expr'
# GOTO lbl [IF expr]    Jump to the specified line. If 'expr' is present,
#                       jump only if the result of 'expr' >= 0
# LET var = expr        Set value of 'var' to 'expr'
#
# There are 33 variables, named X, Y, Z, X0, ..., X9, Y0, ...
# The type of variables and expressions are all 32-bit signed integers
# 
###########################################################################

import sp

class Expr:
    def const_fold (self, evaluator):
        return self

class NumLit (Expr):
    def __init__ (self, num):
        self.num = num
        
    def evaluate (self, interpreter):
        return self.num
    
    def push_to_stack (self, compiler):
        return ['push dword %s' % self.num]        

class VariableVal (Expr):
    def __init__ (self, var):
        self.var = var
        
    def evaluate (self, interpreter):
        return interpreter.get_variable (self.var)

    def push_to_stack (self, compiler):
        return ['push dword [%s]' % compiler.var_address (self.var)]

class BinaryFunction (Expr):
    def __init__ (self, fun, asm_opcode, left, right):
        self.fun = fun
        self.asm_opcode = asm_opcode
        self.left = left
        self.right = right

    def evaluate (self, interpreter):
        return self.fun (self.left.evaluate(interpreter),
                         self.right.evaluate(interpreter))

    def push_to_stack (self, compiler):
        return self.left.push_to_stack (compiler) + \
               ['pop eax'] + \
               self.right.push_to_stack (compiler) + \
               ['pop ebx',
                '%s eax, ebx' % self.asm_opcode,
                'push eax']

    def const_fold (self, evaluator):
        try:
            left = self.left.evaluate (evaluator)
            right = self.right.evaluate (evaluator)
            return NumLit (self.fun (left, right))
        except sp.CannotEvaluateYetError:
            return self

class Plus (BinaryFunction):
    def __init__ (self, left, right):
        BinaryFunction.__init__ (self, lambda x, y: x + y, 'add', left, right)

class Minus (BinaryFunction):
    def __init__ (self, left, right):
        BinaryFunction.__init__ (self, lambda x, y: x - y, 'sub', left, right)

class Mul (BinaryFunction):
    def __init__ (self, left, right):
        BinaryFunction.__init__ (self, lambda x, y: x * y, 'mul', left, right)

class Div (BinaryFunction):
    def __init__ (self, left, right):
        BinaryFunction.__init__ (self, lambda x, y: x / y, 'div', left, right)
